from django.shortcuts import render, redirect, get_object_or_404
from django.contrib.auth import authenticate, login, logout
from django.contrib.auth.decorators import login_required
from django.contrib import messages
from django.urls import reverse
from django.utils import timezone
from .models import Student, Attendance
from .forms import StudentForm, AttendanceDateForm
from django.db import IntegrityError
from datetime import date

# ---------- Auth: simple admin login ----------
def user_login(request):
    if request.user.is_authenticated:
        return redirect('dashboard')

    if request.method == 'POST':
        username = request.POST.get('username')
        password = request.POST.get('password')
        user = authenticate(request, username=username, password=password)
        if user is not None and user.is_staff:
            login(request, user)
            return redirect('dashboard')
        else:
            messages.error(request, 'Invalid credentials or not an admin user.')
    return render(request, 'login.html')


@login_required
def user_logout(request):
    logout(request)
    return redirect('login')

# ---------- Dashboard ----------
@login_required
def dashboard(request):
    total_students = Student.objects.count()
    today = timezone.localdate()
    present_count = Attendance.objects.filter(date=today, status='Present').count()
    absent_count = Attendance.objects.filter(date=today, status='Absent').count()
    context = {
        'total_students': total_students,
        'present_count': present_count,
        'absent_count': absent_count,
        'today': today,
    }
    return render(request, 'dashboard.html', context)

# ---------- Student CRUD ----------
@login_required
def student_list(request):
    students = Student.objects.order_by('roll_number')
    return render(request, 'student_list.html', {'students': students})

@login_required
def add_student(request):
    if request.method == 'POST':
        form = StudentForm(request.POST)
        if form.is_valid():
            form.save()
            messages.success(request, 'Student added successfully.')
            return redirect('student_list')
    else:
        form = StudentForm()
    return render(request, 'student_form.html', {'form': form, 'title': 'Add Student'})

@login_required
def edit_student(request, pk):
    student = get_object_or_404(Student, pk=pk)
    if request.method == 'POST':
        form = StudentForm(request.POST, instance=student)
        if form.is_valid():
            form.save()
            messages.success(request, 'Student updated successfully.')
            return redirect('student_list')
    else:
        form = StudentForm(instance=student)
    return render(request, 'student_form.html', {'form': form, 'title': 'Edit Student'})

@login_required
def delete_student(request, pk):
    student = get_object_or_404(Student, pk=pk)
    if request.method == 'POST':
        student.delete()
        messages.success(request, 'Student deleted.')
        return redirect('student_list')
    return render(request, 'student_confirm_delete.html', {'student': student})

# ---------- Mark Attendance ----------
@login_required
def mark_attendance(request):
    students = Student.objects.all().order_by('roll_number')
    if not students.exists():
        messages.info(request, 'No students found. Please add students first.')
        return redirect('student_list')

    if request.method == 'POST':
        # date selection
        sel_date_str = request.POST.get('date')
        if sel_date_str:
            sel_date = date.fromisoformat(sel_date_str)
        else:
            sel_date = timezone.localdate()

        # Loop through students and read status input
        created = 0
        updated = 0
        for student in students:
            key = f"status_{student.id}"
            status = request.POST.get(key, 'Absent')  # default to Absent if not provided
            # Create or update attendance record
            try:
                obj, created_flag = Attendance.objects.update_or_create(
                    student=student,
                    date=sel_date,
                    defaults={'status': status}
                )
                if created_flag:
                    created += 1
                else:
                    updated += 1
            except IntegrityError:
                # skip duplicates or DB errors
                continue

        messages.success(request, f'Attendance saved. {created} created, {updated} updated for {sel_date}.')
        return redirect('attendance_list')

    else:
        # GET: show form with today's date
        initial_date = timezone.localdate()
        date_form = AttendanceDateForm(initial={'date': initial_date})
        # For nicer UI, fetch existing attendance for this date to pre-select
        existing = Attendance.objects.filter(date=initial_date)
        attendance_map = {a.student_id: a.status for a in existing}
        context = {
            'students': students,
            'attendance_map': attendance_map,
            'date_form': date_form,
        }
        return render(request, 'mark_attendance.html', context)


# ---------- Attendance List / Filter ----------
@login_required
def attendance_list(request):
    # filter by date if provided
    sel_date = request.GET.get('date')
    if sel_date:
        try:
            sel_date_obj = date.fromisoformat(sel_date)
        except ValueError:
            sel_date_obj = timezone.localdate()
    else:
        sel_date_obj = timezone.localdate()

    records = Attendance.objects.filter(date=sel_date_obj).select_related('student').order_by('student__roll_number')
    # optional: show a range / history
    available_dates = Attendance.objects.order_by('-date').values_list('date', flat=True).distinct()[:30]

    context = {
        'records': records,
        'sel_date': sel_date_obj,
        'available_dates': available_dates,
    }
    return render(request, 'attendance_list.html', context)
